/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.jackrabbit.vault.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;

import org.apache.commons.io.input.CloseShieldInputStream;
import org.apache.commons.io.input.TeeInputStream;
import org.apache.jackrabbit.vault.fs.io.MemoryArchive;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An input stream pump feeds a {@link InputStreamPump.Pump} in a dedicated thread with the input read from
 * the given input stream.
 * This is similar to a {@link TeeInputStream} but leverages {@link PipedInputStream} and {@link PipedOutputStream}
 * and can execute additional tasks in the additional thread consuming the PipedInputStream.
 * Only after calling {@link #close()} the PipedInputStream has been fully consumed (as it waits for the pump's thread to complete).
 *
 * @see MemoryArchive
 */
public class InputStreamPump extends InputStream {

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(InputStreamPump.class);

    private final InputStream source;

    private final PipedOutputStream out;

    private final PipedInputStream in;

    private Thread pumpThread;

    @SuppressWarnings("java:S3077") // error is only written from one thread and used as immutable class
    private volatile Exception error;

    public InputStreamPump(InputStream source, final Pump pump) throws IOException {
        this.source = source;

        out = new PipedOutputStream();
        in = new PipedInputStream(out, 8192);

        pumpThread = new Thread(new Runnable() {
            public void run() {
                try {
                    pump.run(new CloseShieldInputStream(in));
                    // ensure that input stream is pumping in case it didn't read to the end
                    byte[] buffer = new byte[8192];
                    while (in.read(buffer) >= 0)
                        ;
                } catch (Exception e) {
                    error = e;
                    log.error("Error while processing input stream", e);
                }
            }
        });
        pumpThread.start();
    }

    public interface Pump {
        /**
         * <p>The specified stream remains open after this method returns.
         * @param in
         * @throws Exception
         */
        void run(InputStream in) throws Exception;
    }

    /**
     *
     * @return exception which has occurred in the pump thread or {@code null}.
     * @deprecated Rather call {@link #close()}, as otherwise this might be called too early (before the thread finished).
     * {@code close()} will automatically wrap the potential exception from the pump in an IOException and throws it as well
     */
    @Deprecated
    public Exception getError() {
        return error;
    }

    @Override
    public int read() throws IOException {
        int b = source.read();
        if (b >= 0) {
            out.write(b);
        }
        return b;
    }

    @Override
    public int read(byte[] b) throws IOException {
        int len = source.read(b);
        if (len > 0) {
            out.write(b, 0, len);
        }
        return len;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        int read = source.read(b, off, len);
        if (read > 0) {
            out.write(b, off, read);
        }
        return read;
    }

    @Override
    public long skip(long n) throws IOException {
        return source.skip(n);
    }

    @Override
    public int available() throws IOException {
        return source.available();
    }

    @Override
    public void close() throws IOException {
        source.close();
        out.flush();
        try {
            out.close();
            pumpThread.join();
            in.close();
        } catch (InterruptedException e) {
            pumpThread.interrupt();
            throw new IOException(e);
        }
        if (error != null) {
            throw new IOException(error);
        }
    }

    @Override
    public synchronized void mark(int readlimit) {
        throw new UnsupportedOperationException("Mark not supported");
    }

    @Override
    public synchronized void reset() throws IOException {
        throw new UnsupportedOperationException("Reset not supported");
    }

    @Override
    public boolean markSupported() {
        return false;
    }
}
