/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.faulttolerance.core.timer;

import io.smallrye.faulttolerance.core.timer.Timer;
import io.smallrye.faulttolerance.core.timer.TimerLogger;
import io.smallrye.faulttolerance.core.timer.TimerTask;
import io.smallrye.faulttolerance.core.util.Preconditions;
import io.smallrye.faulttolerance.core.util.RunnableWrapper;
import java.util.Comparator;
import java.util.NoSuchElementException;
import java.util.SortedSet;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;

public final class ThreadTimer
implements Timer {
    private static final AtomicInteger COUNTER = new AtomicInteger(0);
    private static final Comparator<Task> TASK_COMPARATOR = (o1, o2) -> {
        if (o1 == o2) {
            return 0;
        }
        long delta = o1.startTime - o2.startTime;
        if (delta < 0L) {
            return -1;
        }
        if (delta > 0L) {
            return 1;
        }
        return System.identityHashCode(o1) < System.identityHashCode(o2) ? -1 : 1;
    };
    private final int id;
    private final SortedSet<Task> tasks = new ConcurrentSkipListSet<Task>(TASK_COMPARATOR);
    private final Executor defaultExecutor;
    private final Thread thread;
    private final AtomicBoolean running = new AtomicBoolean(true);

    @Deprecated(forRemoval=true)
    public static ThreadTimer create(Executor defaultExecutor) {
        return new ThreadTimer(defaultExecutor);
    }

    public ThreadTimer(Executor defaultExecutor) {
        this.id = COUNTER.incrementAndGet();
        this.defaultExecutor = Preconditions.checkNotNull(defaultExecutor, "Executor must be set");
        this.thread = new Thread(() -> {
            while (this.running.get()) {
                try {
                    Task task;
                    if (this.tasks.isEmpty()) {
                        LockSupport.park();
                        continue;
                    }
                    try {
                        task = this.tasks.first();
                    }
                    catch (NoSuchElementException e) {
                        continue;
                    }
                    long currentTime = System.nanoTime();
                    long taskStartTime = task.startTime;
                    if (taskStartTime - currentTime <= 0L) {
                        boolean removed = this.tasks.remove(task);
                        if (!removed) continue;
                        Executor executorForTask = task.executor();
                        if (executorForTask == null) {
                            executorForTask = defaultExecutor;
                        }
                        executorForTask.execute(task);
                        continue;
                    }
                    LockSupport.parkNanos(taskStartTime - currentTime);
                }
                catch (Throwable e) {
                    TimerLogger.LOG.unexpectedExceptionInTimerLoop(e);
                }
            }
        }, "SmallRye Fault Tolerance Timer " + this.id);
        this.thread.start();
        TimerLogger.LOG.createdTimer(this.id);
    }

    @Override
    public int getId() {
        return this.id;
    }

    @Override
    public TimerTask schedule(long delayInMillis, Runnable task) {
        return this.schedule(delayInMillis, task, null);
    }

    @Override
    public TimerTask schedule(long delayInMillis, Runnable task, Executor executor) {
        long startTime = System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(delayInMillis);
        task = RunnableWrapper.INSTANCE.wrap(task);
        Task timerTask = executor == null || executor == this.defaultExecutor ? new Task(startTime, task) : new TaskWithExecutor(startTime, task, executor);
        this.tasks.add(timerTask);
        LockSupport.unpark(this.thread);
        TimerLogger.LOG.scheduledTimerTask(timerTask, delayInMillis);
        return timerTask;
    }

    @Override
    public int countScheduledTasks() {
        return this.tasks.size();
    }

    @Override
    public void shutdown() throws InterruptedException {
        if (this.running.compareAndSet(true, false)) {
            TimerLogger.LOG.shutdownTimer(this.id);
            this.thread.interrupt();
            this.thread.join();
        }
    }

    private class Task
    implements TimerTask,
    Runnable {
        final long startTime;
        volatile Runnable runnable;

        Task(long startTime, Runnable runnable) {
            this.startTime = startTime;
            this.runnable = Preconditions.checkNotNull(runnable, "Runnable task must be set");
        }

        @Override
        public boolean isDone() {
            boolean queued = ThreadTimer.this.tasks.contains(this);
            if (queued) {
                return false;
            }
            return this.runnable == null;
        }

        @Override
        public boolean cancel() {
            boolean removed = ThreadTimer.this.tasks.remove(this);
            if (removed) {
                this.runnable = null;
                TimerLogger.LOG.cancelledTimerTask(this);
                return true;
            }
            return false;
        }

        public Executor executor() {
            return null;
        }

        @Override
        public void run() {
            TimerLogger.LOG.runningTimerTask(this);
            try {
                this.runnable.run();
            }
            finally {
                this.runnable = null;
            }
        }
    }

    private final class TaskWithExecutor
    extends Task {
        private final Executor executor;

        TaskWithExecutor(long startTime, Runnable runnable, Executor executor) {
            super(startTime, runnable);
            this.executor = Preconditions.checkNotNull(executor, "Executor must be set");
        }

        @Override
        public Executor executor() {
            return this.executor;
        }
    }
}

