/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.sis.referencing.operation.provider;

import jakarta.xml.bind.annotation.XmlTransient;
import org.opengis.parameter.ParameterDescriptor;
import org.opengis.parameter.ParameterDescriptorGroup;
import org.opengis.parameter.ParameterNotFoundException;
import org.opengis.referencing.cs.EllipsoidalCS;
import org.opengis.referencing.operation.Transformation;
import org.opengis.referencing.operation.MathTransform;
import org.apache.sis.parameter.ParameterBuilder;
import org.apache.sis.parameter.Parameters;
import org.apache.sis.referencing.operation.transform.MathTransforms;
import org.apache.sis.measure.Units;


/**
 * Base class for all transformations that perform a translation in the geographic domain.
 * This base class defines a provider for <q>Geographic3D offsets</q> (EPSG:9660),
 * but subclasses will provide different operations.
 *
 * @author  Martin Desruisseaux (Geomatys)
 */
@XmlTransient
public class GeographicOffsets extends AbstractProvider {
    /**
     * Serial number for inter-operability with different versions.
     */
    private static final long serialVersionUID = -6246011184175753328L;

    /**
     * The operation parameter descriptor for the <q>Longitude offset</q> parameter value.
     *
     * <!-- Generated by ParameterNameTableGenerator -->
     * <table class="sis">
     *   <caption>Parameter names</caption>
     *   <tr><td> EPSG:    </td><td> Longitude offset </td></tr>
     *   <tr><td> EPSG:    </td><td> Longitude rotation </td></tr>
     * </table>
     */
    static final ParameterDescriptor<Double> TX;

    /**
     * The operation parameter descriptor for the <q>Latitude offset</q> parameter value.
     *
     * <!-- Generated by ParameterNameTableGenerator -->
     * <table class="sis">
     *   <caption>Parameter names</caption>
     *   <tr><td> EPSG:    </td><td> Latitude offset </td></tr>
     * </table>
     */
    static final ParameterDescriptor<Double> TY;

    /**
     * The operation parameter descriptor for the <q>Vertical Offset</q> parameter value.
     *
     * <!-- Generated by ParameterNameTableGenerator -->
     * <table class="sis">
     *   <caption>Parameter names</caption>
     *   <tr><td> EPSG:    </td><td> Vertical Offset </td></tr>
     * </table>
     */
    static final ParameterDescriptor<Double> TZ;

    /**
     * The group of all parameters expected by this coordinate operation.
     */
    private static final ParameterDescriptorGroup PARAMETERS;
    static {
        final ParameterBuilder builder = builder();
        TY = builder.addIdentifier("8601").addName("Latitude offset") .create(0, Units.DEGREE);
        TX = builder.addIdentifier("8602").addName("Longitude offset").addName("Longitude rotation").create(0, Units.DEGREE);
        TZ = builder.addIdentifier("8603").addName("Vertical Offset") .create(0, Units.METRE);
        PARAMETERS = builder.addIdentifier("9660").addName("Geographic3D offsets").createGroup(TY, TX, TZ);
    }

    /**
     * The canonical instance of this operation method.
     *
     * @see #provider()
     */
    private static final GeographicOffsets INSTANCE = new GeographicOffsets();

    /**
     * Returns the canonical instance of this operation method.
     * This method is invoked by {@link java.util.ServiceLoader} using reflection.
     *
     * @return the canonical instance of this operation method.
     */
    public static GeographicOffsets provider() {
        return INSTANCE;
    }

    /**
     * Creates a new provider.
     *
     * @todo Delete this constructor after we stop class-path support,
     *       replaced by direct call to the constructor below.
     */
    public GeographicOffsets() {
        this(PARAMETERS, (byte) 3);
    }

    /**
     * For default constructors in this class and subclasses.
     *
     * @param parameters  description of parameters expected by this operation.
     * @param dimension   number of source dimensions (2 or 3).
     */
    GeographicOffsets(ParameterDescriptorGroup parameters, byte dimension) {
        super(Transformation.class, parameters,
              EllipsoidalCS.class, false,
              EllipsoidalCS.class, false,
              dimension);
    }

    /**
     * Returns the parameter descriptor for the vertical axis.
     * This parameter is different in the case of {@link GeographicAndVerticalOffsets}.
     */
    ParameterDescriptor<Double> vertical() {
        return TZ;
    }

    /**
     * The inverse of {@code GeographicOffsets} is the same operation with parameter signs inverted.
     *
     * @return {@code this}.
     */
    @Override
    public final AbstractProvider inverse() {
        return this;
    }

    /**
     * Returns the operation method which is the closest match for the given transform.
     * This is an adjustment based on the number of dimensions only, on the assumption
     * that the given transform has been created by this provider or a compatible one.
     */
    @Override
    public AbstractProvider variantFor(final MathTransform transform) {
        if (getClass() == GeographicOffsets.class && maxDimension(transform) < 3) {
            return GeographicOffsets2D.provider();
        }
        return this;
    }

    /**
     * Creates a transform from the specified group of parameter values.
     * The parameter values are unconditionally converted to degrees and metres.
     *
     * @param  context  the parameter values together with its context.
     * @return the created math transform.
     * @throws ParameterNotFoundException if a required parameter was not found.
     */
    @Override
    public MathTransform createMathTransform(final Context context) {
        final Parameters pv = Parameters.castOrWrap(context.getCompletedParameters());
        return MathTransforms.translation(pv.doubleValue(TX),
                                          pv.doubleValue(TY),
                                          pv.doubleValue(vertical()));
    }
}
